﻿/*
 * --------------------------------------
 * sazameki -- audio manipulating library
 * http://sazameki.org/
 * --------------------------------------
 * 
 * - developed by     Takaaki Yamazaki
 *                    http://www.zkdesign.jp/
 * - supported by     Spark project
 *                    http://www.libspark.org/
 */

/*
 * Licensed under the MIT License
 * 
 * Copyright (c) 2008 Takaaki Yamazaki
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */



/**
* 文字列をパースして処理の準備をするためのクラス。
* @author Takaaki Yamazaki
* @version 0.1
*/

package org.sazameki.audio.engine.ssGenerator.engine {
	import org.sazameki.audio.core.SoundFactory;
	import org.sazameki.audio.engine.ssGenerator.SsAudioSetting;
	import org.sazameki.audio.events.AudioEvent;
	import org.sazameki.audio.events.AudioProgressEvent;
	
	import flash.events.Event;
	import flash.events.EventDispatcher;
	import flash.media.Sound;
	
	import org.sazameki.audio.engine.ssGenerator.processor.IMasterProcessor;
	import org.sazameki.audio.engine.ssGenerator.processor.MasterProcessor;
	import org.sazameki.audio.processor.ISsAudioProcessor;

	public class EngineBase extends EventDispatcher implements ISsEngine {

		//処理を受け持つクラス
		protected var processor:IMasterProcessor;
		protected var setting:SsAudioSetting;
		protected var audioProcessors:Object={};
		protected var samplesPerUpdate:int;
		protected var wannaGenSoundObj:Boolean;
		protected var factory:SoundFactory;

		/**
		 * Constructor.
		 * @param	processSamplesPerUpdate
		 * @param	optionalSoundFactory
		 */
		public function EngineBase(processSamplesPerUpdate:int=0,optionalSoundFactory:SoundFactory=null) {
			samplesPerUpdate=processSamplesPerUpdate;
			if(optionalSoundFactory){
				factory=optionalSoundFactory;
			}else{
				factory=new SoundFactory();
			}
		}
		
		/**
		 * サウンド定義文字列をパースする
		 * @param	str
		 */
		public function parse(str:String):void { 
			
			//オーディオの設定をする（s-がなかった場合のため）
			setupAudio();

			//改行文字、空白文字を除去
			var pattern:RegExp = /[ \n\r]/g;
			str=str.replace(pattern, '');
			
			//processor生成
			processor=parseString(str,createProcessor('master'));
		}
		
		/**
		 * 
		 * @param	str
		 * @param	proc
		 * @return
		 */
		protected function parseString(str:String,proc:IMasterProcessor):IMasterProcessor{
			
		
			//最後にカンマついてなかったらカンマつける。
			if(str.charAt(-1) != ','){
				str+=',';
			}

			//分割（カンマ区切り/波カッコ内のカンマは無視）
			var pattern:RegExp=/[^,]+?:\{.+?\},|[^,]+?:.+?,/g;
			var result:String;

			var id:String;
			var idx:int;
			var arr:Array;
			var tmpPattern:RegExp = /,$/;
			while ((result = pattern.exec(str))) {
				
				//行末のカンマを削除
				result=result.replace(tmpPattern,'');
				
				//最初の:より前の部分を抜き出してidに割り当て
				idx=result.indexOf(':');
				id=result.substring(0,idx);
				result=result.substring(idx+1);
				
				//setting指定(s:)だったら
				if(id=='s'){
					arr=result.split('-');
					setupAudio(int(arr[0]),int(arr[1]),int(arr[2]));
				}

				if (result.charAt(0)!='{') {
					//<波カッコがない場合>
					
					//「+」がある場合はさらに分割してsubProcessor作る
					if(result.indexOf('+') != -1){
						arr=result.split('+');
						result=arr.shift();

						//subProcessor（要するに各SsAudioProcessor内で利用するSsAudioProcessor)作る
						var len:int=arr.length;
						var obj:Object=new Object();
						for(var i:int=0;i<len;i++){
							arr[i]=String(arr[i]).split('-');
							obj[arr[i].shift()]=getAudioProcessor(arr[i].shift(),arr[i]);
						}
						
						//SsAudioProcessor作る
						proc.add(getAudioProcessor(id,result.split('-'),obj));
						
					}else{
						//でなければ素直に追加
						proc.add(getAudioProcessor(id,result.split('-')));
					}
					
					
				}else {
					//波カッコがある場合
					//波カッコ削除
					result=result.replace(/^\{/,'');
					result=result.replace(/\}$/,'');
					//新しいProcessorを作ってパース(再帰処理)
					proc.addSubProcessor(id,parseString(result,createProcessor(id)));
				}
			}
			
			return proc;
		}
		

		
		//初期化処理
		protected function setupAudio(channels:int=2,rate:int=44100,bit:int=16 ):void {
			setting = new SsAudioSetting(channels, rate, bit,samplesPerUpdate);
		}
		
		protected function createProcessor(id:String):IMasterProcessor{
			var p:IMasterProcessor=new MasterProcessor(id);
			p.initialize(setting);
			return(p);
		}

		
		protected function getAudioProcessor(id:String,params:Array,subProcessors:Object=null):ISsAudioProcessor{
			var cls:Class = audioProcessors[id];
			if(cls){
				var obj:ISsAudioProcessor=new cls();
				obj.initialize(setting,params,subProcessors);
				return obj;
			}else{
				return null;
			}
		}
		
		
		//処理開始
		public function process(wannaGenerateSoundObject:Boolean=true):void {
			wannaGenSoundObj=wannaGenerateSoundObject;
			processor.addEventListener(Event.COMPLETE, onProcessComplete);
			processor.start();
		}

		protected function onProcessComplete(e:Event) :void{
			dispatchEvent(new AudioProgressEvent(AudioProgressEvent.ON_DATA,processor.result,setting));
			if(wannaGenSoundObj){
				factory.addEventListener(AudioEvent.COMPLETE,onSoundCreationComplete);
				factory.generateSound(processor.result,setting);
			}
			processor.removeEventListener(Event.COMPLETE,onProcessComplete);
		}
		
		protected function onSoundCreationComplete(e:AudioEvent):void {
			dispatchEvent(new AudioEvent(AudioEvent.COMPLETE,e.sound));
			factory.removeEventListener(AudioEvent.COMPLETE,onSoundCreationComplete);
		}
		
		
		
		
		
		
		//Utility
		//文字列からprocessor名候補を返す
		public function getCandidate(str:String,isEnvelope:Boolean=false):Array{
			var arr:Array=new Array();
			var strLen:int=str.length;
			var objname:String;
			if(isEnvelope){
				for(objname in audioProcessors){
					if(objname.substr(0,strLen)==str){
						var a:ISsAudioProcessor=new (audioProcessors[objname] as Class)();
						if(a.isEnvelope){
							arr.push(objname);
						}
					}
				}
			}else{
				for(objname in audioProcessors){
					if(objname.substr(0,strLen)==str){
						arr.push(objname);
					}
				}
			}
			return(arr);
		}
		//文字列からprocessorのdescriptionとparameterFormatを返す
		public function getFormatString(str:String):Array{
			var arr:Array=new Array();
			var objname:String;
			for(objname in audioProcessors){
				if(objname==str){
					var a:ISsAudioProcessor=new (audioProcessors[objname] as Class)();
					arr.push(a.description);
					arr.push(a.parameterFormat);
					return(arr);
				}
			}
			return(null);
		}
	}
}
