﻿/*
 * --------------------------------------
 * sazameki -- audio manipulating library
 * http://sazameki.org/
 * --------------------------------------
 * 
 * - developed by     Takaaki Yamazaki
 *                    http://www.zkdesign.jp/
 * - supported by     Spark project
 *                    http://www.libspark.org/
 */

package org.sazameki.audio.processor.filter {
	import org.sazameki.audio.core.Sample;
	import org.sazameki.audio.engine.ssGenerator.SsAudioSetting;
	import org.sazameki.audio.processor.envelope.IEnvelope;
	import org.sazameki.audio.processor.SsAudioProcessor;

	/**
	 * Moog-like VCF(LPF with Envelope)
	 * algorithm : http://www.musicdsp.org/archive.php?classid=3#24
	 * @author Takaaki Yamazaki
	 */
	public class MoogVCF extends SsAudioProcessor {
		
		private var _resonance:Number;
		private var _cutoff:Number;
		private var _cutoffEnv:IEnvelope;
		private var _minFreq:Number;
		private var _maxFreq:Number;
		private var _freqDiff:Number;
		
		private var k:Number;
		private var p:Number;
		private var r:Number;

		
		private var oldlx:Number;
		private var oldrx:Number;
		
		
		private var l1:Number,l2:Number,l3:Number,l4:Number;
		private var oldl1:Number,oldl2:Number,oldl3:Number;
		private var r1:Number,r2:Number,r3:Number,r4:Number;
		private var oldr1:Number,oldr2:Number,oldr3:Number;
		

		override public function initialize(setting:SsAudioSetting, basicParams:Array = null, additionalProcessors:Object = null):void{
			super.initialize(setting, basicParams, additionalProcessors);
			
			_minFreq=basicParams[0];
			_maxFreq=basicParams[1];
			_freqDiff=_maxFreq-_minFreq;
			_resonance=basicParams[2];
			_cutoffEnv=additionalProcessors.env;
		}
		override public function get description():String{
			return "Voltage Controlled Filter."
		}
		override public function get parameterFormat():String{
			return 'min frequency(Hz)-max frequency(Hz)-resonance(0-1)+env-[envelope]'
		}
		override public function trigger():void{
			super.trigger();
			
			l1=l2=l3=l4=oldl1=oldl2=oldl3=0;
			r1=r2=r3=r4=oldr1=oldr2=oldr3=0;
			oldlx=oldrx=0;
		}
		
		
		
		public function setCutoff(frequencyHz:Number):void{
			_cutoff=frequencyHz;
			calc();
		}
		
		private function calc():void{
			var f:Number;
			var scale:Number;
			
			f=2*_cutoff/_setting.sampleRate;
			
			k=3.6*f - 1.6*f*f -1;
			p=(k+1)*0.5;
			scale=Math.pow(Math.E,(1-p)*1.386249);
			
			r=_resonance*scale;
			
		}
		
		
		override public function processAudio(samples:Array):void{
			
			var len:int=samples.length;
			var smpl:Sample;
			var inputL:Number;
			var inputR:Number;
			var x:Number;
			
			
			for(var i:int=0;i<len;i++){
				
				//updateFrequency
				if(_cutoffEnv){
					setCutoff(_cutoffEnv.getEnvAt(_idx)*_freqDiff+_minFreq);
				}
				
				//process
				smpl=samples[i];
				
				inputL=smpl.left;
				inputR=smpl.right;
				
				
				//process Lch
				
				
				x = inputL - r*l4;

				l1=x*p + oldlx*p - k*l1;
				l2=l1*p+oldl1*p - k*l2;
				l3=l2*p+oldl2*p - k*l3;
				l4=l3*p+oldl3*p - k*l4;

				l4 = l4 - (Math.pow(l4,3))/6;

				oldlx = x;
				oldl1 = l1;
				oldl2 = l2;
				oldl3 = l3;
				
				
				//process Rch

				x = inputR - r*r4;

				r1=x*p + oldrx*p - k*r1;
				r2=r1*p+oldr1*p - k*r2;
				r3=r2*p+oldr2*p - k*r3;
				r4=r3*p+oldr3*p - k*r4;

				r4 = r4 - (Math.pow(r4,3))/6;

				oldrx = x;
				oldr1 = r1;
				oldr2 = r2;
				oldr3 = r3;
				
				//
				
				smpl.left=l4;
				smpl.right=r4;
				
				_idx++;
			}
		}
		
	}

}
