﻿package org.sazameki.audio.processor.filter 
{
	import org.sazameki.audio.core.AudioSamples;
	import org.sazameki.audio.parameter.Parameter;
	import org.sazameki.audio.parameter.ParameterExponential;
	import org.sazameki.audio.processor.IAudioProcessor;
	
	/**
	* Resonant Filter
	* http://www.musicdsp.org/showArchiveComment.php?ArchiveID=29
	* @author Takaaki Yamazaki(zk design)
	*/
	public class ResonatFilter implements IAudioProcessor
	{
		private var _filterType:int;
		private var _paramCutoff:ParameterExponential = new ParameterExponential(100,16000);
		private var _paramQ:Parameter=new Parameter(0,1);
		
		/**
		 * constructor
		 * @param	type LPF=0,BPF=1,HPF=2
		 * @param	cutoff
		 * @param	q
		 */
		public function ResonatFilter(type:int,cutoff:Number=440,q:Number=0.5) 
		{
			_filterType = type;
			_paramCutoff.value = cutoff;
			_paramQ.value = q;
		}

		
		private var _bufL0:Number = 0 ;
		private var _bufL1:Number = 0 ;
		private var _bufR0:Number = 0 ;
		private var _bufR1:Number = 0;
		
		public function process(samples:AudioSamples):void
		{
			
			var len:int = samples.length;
			var cutOffFreq:Number = _paramCutoff.value;
			var q:Number = _paramQ.value;
			
			
			// f and fb calculation
			var f:Number = 2.0*Math.sin(Math.PI*cutOffFreq/samples.setting.sampleRate);
			/* you can approximate this with f = 2.0*pi*freq/samplerate with tuning error towards nyquist */
			var fb:Number = q + q/(1.0 - f);

			var i:int = 0;
			var left:Vector.<Number>=samples.left;
			var hp:Number;
			var bp:Number;
			for (; i < len; ++i)
			{
				hp = left[i] - _bufL0;
				bp = _bufL0 - _bufL1;
				_bufL0 = _bufL0 + f * (hp + fb * bp);
				_bufL1 = _bufL1 + f * (_bufL0 - _bufL1);

				left[i]=(_filterType==0)?_bufL1:(_filterType==1)?bp:hp;
			}

			if (samples.setting.channels == 2)
			{
				var right:Vector.<Number>=samples.right;
				for (i = 0; i < len; ++i)
				{
					hp = right[i] - _bufR0;
					bp = _bufR0 - _bufR1;
					_bufR0 = _bufR0 + f * (hp + fb * bp);
					_bufR1 = _bufR1 + f * (_bufR0 - _bufR1);

					right[i]=(_filterType==0)?_bufR1:(_filterType==1)?bp:hp;

				}
			}
		}
		
		public function get cutoff():ParameterExponential { return _paramCutoff; }
		public function get q():Parameter { return _paramQ; }
		
	}
	
}