﻿/*
 * --------------------------------------
 * sazameki -- audio manipulating library
 * http://sazameki.org/
 * --------------------------------------
 * 
 * - developed by     Takaaki Yamazaki
 *                    http://www.zkdesign.jp/
 * - supported by     Spark project
 *                    http://www.libspark.org/
 */

/*
 * Licensed under the MIT License
 * 
 * Copyright (c) 2008 Takaaki Yamazaki
 * 
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 * 
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 * 
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */
package org.sazameki.audio.processor.oscillator {
	import org.sazameki.audio.core.Sample;
	import org.sazameki.audio.core.AudioSetting;
	import org.sazameki.audio.core.SampleBuffer;
	import org.sazameki.audio.engine.ssGenerator.SsAudioSetting;
	import org.sazameki.audio.processor.AudioProcessor;
	import org.sazameki.audio.processor.envelope.IEnvelope;
	import org.sazameki.audio.processor.SsAudioProcessor;
	import org.sazameki.audio.utils.Converter;

	public class Oscillator extends AudioProcessor{
		
		
		protected var _minFreq:Number=440;
		protected var _maxFreq:Number=440;
		protected var diffFreq:Number;
		protected var phase:Number;
		protected var _freq:Number;
		protected var phaseModCoeff:Number;
		
		
		protected var pitchEnvelope:IEnvelope;
		protected var ampEnvelope:IEnvelope;
		
		private var conv:Converter=new Converter();
		private var noteArr:Array=["C","C#","D","D#","E","F","F#","G","G#","A","A#","B"];
		
		
		override public function get parameterFormat():String{
			return('min frequency(Hz)-[max frequency(Hz)]-[frequency modulation by input(0-10000)](+penv-[pitch envelope])(+aenv-[amplitude envelope])')
		}
		
		public function set minFreq(value:Number):void 
		{
			_minFreq = value;
			
		}
		
		public function set freq(value:Number):void 
		{
			_minFreq = value;
			_maxFreq = value;
		}
		
		override public function initialize(setting:AudioSetting,params:Array=null,additionalProcessors:Object=null):void{
			super.initialize(setting,params);
			phase=0;
			
			//params:ピッチの設定(min,max)
			if(params){
				_minFreq=getFreq(params[0]);
				if(params[1]){
					_maxFreq=getFreq(params[1]);
				}else{
					_maxFreq=_minFreq;
				}
				freq=_minFreq;
				diffFreq=_maxFreq-_minFreq;

				//phase modulationする？
				if(params[2]){
					phaseModCoeff=params[2];
				}
			}
			
			//pitch envelope

			if(additionalProcessors){
				if(additionalProcessors.penv){
					pitchEnvelope=additionalProcessors.penv;
				}
				
				//amplitude envelope
				if(additionalProcessors.aenv){
					ampEnvelope=additionalProcessors.aenv;
				}
			}
			
			
		}
		
		private function getFreq(str:String):Number{
			if(str.charAt(0)=='m'){
				//MIDI NOTE Num
				str=str.substr(1);
				return(conv.MidiNoteToFreq440(int(str)));
			}else if(isNaN(Number(str))){
				//C0とかD#0とか
				var note:String;
				var oct:int;
				if(str.charAt(1)=='#'){
					note=str.substr(0,2);
					oct=int(str.substr(2));
				}else{
					note=str.substr(0,1);
					oct=int(str.substr(1));
				}
				var midiNum:int=noteArr.indexOf(note)+(oct*12)+12;
				return(conv.MidiNoteToFreq440(midiNum));
			}else{
				return(Number(str));
			}
		}
		
		public function generate(phase:Number):Number{
			//require overriden
			return 0;
		}
		
		
		public function processAudio(samples:SampleBuffer):void{
			
			//サンプル生成
			var len:int = samples.length;
			
			
			var bufferL:Vector.<Number> = samples.left;
			var bufferR:Vector.<Number> = samples.right;
			var hasRight:Boolean = Boolean(bufferR);
			var sig:Number;
			var c:Number=Math.PI*2/_setting.sampleRate;
			var input:Number;
			
			for(var i:int=0;i<len;i++){
				
				input = bufferL[i];
				sig=generate(phase);
				
				if(ampEnvelope){
					sig=sig*ampEnvelope.getEnvAt(_idx);
				}
				
				
				//波形とボリュームのエンベロープ適用
				bufferL[i] = sig;
				if(hasRight){ bufferR[i] = sig }
				
				//ピッチエンベロープ取得して今のピッチ設定
				if(pitchEnvelope){
					_freq=_minFreq+(diffFreq*pitchEnvelope.getEnvAt(_idx));
				}else{
					_freq=_minFreq;
				}
				
				//phase modulation
				if(phaseModCoeff){
					_freq+=input*phaseModCoeff;
				}
				
				
				//phaseの加算
				phase+=c*_freq;

				_idx++;
			}
			
		}
		
		
	}
	
}
