class App


  constructor: ->
    @initScene()
    @initObjects()
    @animate()


  initScene: =>
    @renderer = new THREE.WebGLRenderer(clearColor: 0x333333, clearAlpha: 1)
    if not @renderer.supportsFloatTextures() or not @renderer.supportsVertexTextures()
      console.log 'float texture:', @renderer.supportsFloatTextures()
      console.log 'vertex texture:', @renderer.supportsVertexTextures()
      Detector.addGetWebGLMessage()
      return

    @renderer.setSize(window.innerWidth, window.innerHeight)

    @camera = new THREE.OrthographicCamera(0, window.innerWidth, 0, window.innerHeight, 1, 1000)
    @camera.position.z = 500

    @scene = new THREE.Scene()

    container = document.getElementById('container')
    container.appendChild(@renderer.domElement)

    @params =
      numColors: 10
      attractivePower: 15
      attractiveRadius: 250
      repulsivePower: 15
      repulsiveRadius: 50
      particleSize: 6
      maxForce: 1
      restart: =>
        @initData()
        @updatePass.uniforms.reset.value = 1
        @particles.material.uniforms.kind.value = @updatePass.uniforms.tKind.value

    @gui = new dat.GUI()
    @gui.add(@params, 'numColors', 1, 30).name('Num colors').step(1)
    @gui.add(@params, 'attractivePower', 0, 200).name('Attractive power').onChange (value) =>
      @updatePass.uniforms.attractivePower.value = value
    @gui.add(@params, 'attractiveRadius', 1, 300).name('Attractive radius').onChange (value) =>
      @updatePass.uniforms.attractiveRadius.value = value
    @gui.add(@params, 'repulsivePower', 0, 200).name('Repulsive power').onChange (value) =>
      @updatePass.uniforms.repulsivePower.value = value
    @gui.add(@params, 'repulsiveRadius', 1, 100).name('Repulsive radius').onChange (value) =>
      @updatePass.uniforms.repulsiveRadius.value = value
    @gui.add(@params, 'maxForce', 0.1, 10.0).name('Max force').onChange (value) =>
      @updatePass.uniforms.maxForce.value = value
    @gui.add(@params, 'particleSize', 1, 32).name('Particle size').onChange (value) =>
      @particles.material.uniforms.particleSize.value = value
    @gui.add(@params, 'restart').name('Restart')

    @stats = new Stats()
    container.appendChild(@stats.domElement)

    window.addEventListener 'resize', =>
      @camera.left = 0
      @camera.right = window.innerWidth
      @camera.top = 0
      @camera.bottom = window.innerHeight
      @camera.updateProjectionMatrix()
      @renderer.setSize(window.innerWidth, window.innerHeight)
      @updatePass.uniforms.screenSize.value.set(window.innerWidth, window.innerHeight)

    window.addEventListener 'mousemove', (e) =>
      @updatePass.uniforms.mouse.value.set(e.clientX, e.clientY)

    window.addEventListener 'mousedown', (e) =>
      if e.target is @renderer.domElement
        @updatePass.uniforms.mousePower.value = 10000

    window.addEventListener 'mouseup', =>
      @updatePass.uniforms.mousePower.value = 0

    window.addEventListener 'deviceorientation', (e) =>
      g = @updatePass.uniforms.gravity.value
      g.x += (e.gamma / 30 - g.x) * .1
      g.y += (e.beta / 30 - g.y) * .1


  initObjects: ->
    @dataWidth = 64
    @dataHeight = 64
    options =
      magFilter: THREE.NearestFilter
      minFilter: THREE.NearestFilter
      format: THREE.RGBAFormat
      type: THREE.FloatType
      depthBuffer: false
      stencilBuffer: false
      generateMipmaps: false
    target = new THREE.WebGLRenderTarget(@dataWidth, @dataHeight, options)
    @updater = new THREE.EffectComposer(@renderer, target)
    @updatePass = new THREE.ShaderPass(
      uniforms:
        reset: type: 'f', value: 1
        dataSize: type: 'v2', value: new THREE.Vector2(@dataWidth, @dataHeight)
        screenSize: type: 'v2', value: new THREE.Vector2(window.innerWidth, window.innerHeight)
        mouse: type: 'v2', value: new THREE.Vector2()
        gravity: type: 'v2', value: new THREE.Vector2()
        attractivePower: type: 'f', value: @params.attractivePower
        attractiveRadius: type: 'f', value: @params.attractiveRadius
        repulsivePower: type: 'f', value: @params.repulsivePower
        repulsiveRadius: type: 'f', value: @params.repulsiveRadius
        mousePower: type: 'f', value: 0
        maxForce: type: 'f', value: @params.maxForce
        tDiffuse: type: 't', value: null
        tInitial: type: 't', value: null
        tKind: type: 't', value: null
      vertexShader: """
        varying vec2 vUv;
        void main() {
          gl_Position = projectionMatrix * modelViewMatrix * vec4(position, 1.);
          vUv = uv;
        }
        """
      fragmentShader: """
        uniform float reset;
        uniform vec2 dataSize;
        uniform vec2 screenSize;
        uniform vec2 mouse;
        uniform vec2 gravity;
        uniform float attractivePower;
        uniform float attractiveRadius;
        uniform float repulsivePower;
        uniform float repulsiveRadius;
        uniform float mousePower;
        uniform float maxForce;
        uniform sampler2D tDiffuse;
        uniform sampler2D tInitial;
        uniform sampler2D tKind;

        varying vec2 vUv;

        float rand(vec2 co) {
          return fract(sin(dot(co.xy, vec2(12.9898, 78.233))) * 43758.5453);
        }
        void main() {
          vec4 p;
          if (reset > 0.) {
            p = texture2D(tInitial, vUv);
          } else {
            p = texture2D(tDiffuse, vUv);
            vec2 f = gravity;

            vec3 kind = texture2D(tKind, vUv).rgb;
            for (int y = 0; y < int(dataSize.y); ++y) {
              for (int x = 0; x < int(dataSize.x); ++x) {
                vec4 node = texture2D(tDiffuse, vec2(x, y) / dataSize);
                vec3 kind2 = texture2D(tKind, vec2(x, y) / dataSize).rgb;
                vec2 d = node.xy - p.xy;
                float dist = length(d);
                if (1. < dist && dist < repulsiveRadius) {
                  f -= d / pow(dist, 3.) * repulsivePower;
                } else if (repulsiveRadius <= dist && dist < attractiveRadius && kind == kind2) {
                  f += d / pow(dist, 3.) * attractivePower;
                }
                if (1. < dist && kind == kind2) {
                  float rad = atan(node.w, node.z);
                  f.x += cos(rad) / dist / dist;
                  f.y += sin(rad) / dist / dist;
                }
              }
            }

            if (mousePower > 0.) {
              vec2 mf = mouse - p.xy;
              float mdistsq = mf.x * mf.x + mf.y * mf.y;
              float mrad = atan(mf.y, mf.x);
              if (mdistsq > 1.) {
                f.x -= cos(mrad) / mdistsq * mousePower;
                f.y -= sin(mrad) / mdistsq * mousePower;
              }
            }

            if (length(f) > maxForce) {
              f = normalize(f) * maxForce;
            }

            f.x += (rand(p.xy) - 0.5) * .2;
            f.y += (rand(p.zx) - 0.5) * .2;

            p.zw = p.zw * .9 + f;
            p.xy += p.zw;

            if (p.x < 0.) {
              p.x = -p.x;
              p.z *= -1.;
            } else if (p.x > screenSize.x) {
              p.x = screenSize.x - (p.x - screenSize.x);
              p.z *= -1.;
            }
            if (p.y < 0.) {
              p.y = -p.y;
              p.w *= -1.;
            } else if (p.y > screenSize.y) {
              p.y = screenSize.y - (p.y - screenSize.y);
              p.w *= -1.;
            }
          }
          gl_FragColor = p;
        }
        """
    )
    @initData()
    @updater.addPass(@updatePass)

    geometry = new THREE.Geometry()
    for y in [0...@dataHeight]
      for x in [0...@dataWidth]
        geometry.vertices.push(new THREE.Vector3(x, y, 0))
    material = new THREE.ShaderMaterial
      depthTest: false
      depthWrite: false
      transparent: true
      blending: THREE.AdditiveBlending
      uniforms:
        data: type: 't', value: null
        dataSize: type: 'v2', value: new THREE.Vector2(@dataWidth, @dataHeight)
        map: type: 't', value: @generateNodeTex()
        kind: type: 't', value: @updatePass.uniforms.tKind.value
        particleSize: type: 'f', value: @params.particleSize
      vertexShader: """
        uniform sampler2D data;
        uniform vec2 dataSize;
        uniform sampler2D kind;
        uniform float particleSize;
        varying vec3 vColor;
        void main() {
          vec4 p = texture2D(data, position.xy / dataSize);
          gl_Position = projectionMatrix * modelViewMatrix * vec4(p.x, p.y, 0., 1.);
          gl_PointSize = particleSize;
          vColor = texture2D(kind, position.xy / dataSize).rgb;
        }
        """
      fragmentShader: """
        uniform sampler2D map;
        varying vec3 vColor;
        void main() {
          gl_FragColor = texture2D(map, gl_PointCoord);
          gl_FragColor.rgb *= vColor;
          gl_FragColor.a = 0.7;
        }
        """
    @particles = new THREE.ParticleSystem(geometry, material)
    @scene.add(@particles)


  initData: =>
    data = new Float32Array(4 * @dataWidth * @dataHeight)
    for i in [0...@dataWidth * @dataHeight]
      data[i * 4] = Math.random() * window.innerWidth
      data[i * 4 + 1] = Math.random() * window.innerHeight
      data[i * 4 + 2] = Math.random() * 3 - 1.5
      data[i * 4 + 3] = Math.random() * 3 - 1.5
    initPosition = new THREE.DataTexture(data, @dataWidth, @dataHeight, THREE.RGBAFormat, THREE.FloatType)
    initPosition.minFilter = THREE.NearestFilter
    initPosition.magFilter = THREE.NearestFilter
    initPosition.needsUpdate = true
    @updatePass.uniforms.tInitial.value?.dispose()
    @updatePass.uniforms.tInitial.value = initPosition

    data = new Uint8Array(3 * @dataWidth * @dataHeight)
    colors = []
    for i in [0..@params.numColors]
      c = new THREE.Color()
      c.setHSL(i / @params.numColors, 1.0, 0.5)
      colors.push(c)
    for i in [0...@dataWidth * @dataHeight]
      c = colors[(Math.random() * @params.numColors) | 0]
      data[i * 3] = c.r * 0xff
      data[i * 3 + 1] = c.g * 0xff
      data[i * 3 + 2] = c.b * 0xff
    kind = new THREE.DataTexture(data, @dataWidth, @dataHeight, THREE.RGBFormat)
    kind.minFilter = THREE.NearestFilter
    kind.magFilter = THREE.NearestFilter
    kind.needsUpdate = true
    @updatePass.uniforms.tKind.value?.dispose()
    @updatePass.uniforms.tKind.value = kind


  generateNodeTex: ->
    canvas = document.createElement('canvas')
    canvas.width = 32
    canvas.height = 32
    ctx = canvas.getContext('2d')
    ctx.clearRect(0, 0, 32, 32)
    ctx.beginPath()
    ctx.arc(16, 16, 15, 0, Math.PI * 2, false)
    ctx.fillStyle = 'white'
    ctx.fill()
    tex = new THREE.Texture(canvas)
    tex.needsUpdate = true
    return tex


  animate: =>
    requestAnimationFrame(@animate)

    @updater.render()
    @particles.material.uniforms.data.value = @updater.renderTarget1
    tmp = @updater.renderTarget1
    @updater.renderTarget1 = @updater.renderTarget2
    @updater.renderTarget2 = tmp

    @renderer.render(@scene, @camera)

    @updatePass.uniforms.reset.value = 0

    @stats.update()


if Detector.webgl
  new App()
else
  Detector.addGetWebGLMessage()
