package
{
	import nu.mine.flashnet.sound.core.*;
	import nu.mine.flashnet.sound.synthesis.ToneGenerator;
	/**
	 * An example class demonstrating how to make an AudioDataProvider.
	 * Things to note:
	 * extends AudioDataProvider
	 * creates a PCMSoundFormat and passes it to super(fmt)
	 * overrides fillBuffer
	 * writes mono samples one at a time to buffer, with call buffer.writeMonoSample
	 * @author spender
	 * 
	 */
	public class SineWaveProvider extends AudioDataProvider
	{
		private var generator:ToneGenerator;
		private var freq:Number;
		private var prevFreq:Number;

		/**
		 * creates a SineWaveProvider.
		 * subclasses of AudioDataProvider must always call super(fmt) with a PCMSoundFormat object
		 * @return 
		 * 
		 */
		public function SineWaveProvider()
		{
			var fmt:PCMSoundFormat=new PCMSoundFormat(44100,16,1); //define a format of 44100hZ sample rate, 16 bit audio, 1 channel
			super(fmt);
			this.freq = this.prevFreq = 440;

			this.generator = ToneGenerator.getGenerator(fmt, ToneGenerator.TRIANGULAR);
			this.generator.setPitch(this.freq);
		}
		/**
		 * this is where the audio is generated. Values should lie between -1 and +1 to avoid clipping.
		 * every sample must be stored in the buffer with buffer.writeMonoSample(val)
		 * this function should write numSamples*soundFormat.channels monophonic samples into the buffer
		 * if soundFormat.channels is 2, then the channels are interleaved, so each consecutive call to buffer.writeMonoSample will store a samples for left then right channels etc...
		 * @param numSamples The number of complete samples that must be written (for a stereo format, this means 2 monophonic samples per sample)
		 * 
		 */
		protected override function fillBuffer(numSamples:int):void
		{
			var i:int;
			if (this.freq != this.prevFreq) {
				var df:Number = this.freq - this.prevFreq;
				for(i = 0; i < numSamples; ++i) {
					this.buffer.writeMonoSample(this.generator.getNextSample());
					//trace(this.freq, this.prevFreq, this.prevFreq + df * i / numSamples);
					this.generator.setPitch(this.prevFreq + df * i / numSamples);
				}
				this.prevFreq = this.freq;
				
			} else {
				for(i = 0 ; i < numSamples; ++i) {
					this.buffer.writeMonoSample(this.generator.getNextSample());
				}
			}
			
		}
		
		public function setPitch(freq:Number):void {
			if (this.freq == this.prevFreq) {
				this.prevFreq = this.freq;
			}
			this.freq = freq;
		}
		
		
	}

}