package nu.mine.flashnet.sound.core
{
	import flash.events.EventDispatcher;
	import flash.events.Event;
	import flash.media.Sound;
	import flash.utils.Timer;
	import flash.events.TimerEvent;
	import flash.media.SoundChannel;
	import nu.mine.flashnet.debug.Debug;

	/**
	 * The playback engine. Responsible for playing Sound objects end-to-end in a seamless fashion
	 * @author spender
	 * 
	 */
	public class PlaybackEngine extends EventDispatcher
	{
		/**
		* Constant representing the READY event emitted by this class
		*/
		public static const READY:String="engineReady";
		private var factory:SoundFactory;
		private var soundStack:Array;
		private var ready:Boolean;
		private var silentSound:Sound;
		private var dataProvider:IAudioDataProvider;
		private var playbackBufferSize:int;
		private var newBufferSize:int;
		private var changingBufferSize:Boolean;
		/**
		 * Instantiates a PlaybackEngine.
		 * @param dataProvider The source of the data to be played
		 * @param playbackBufferSize The buffer size, in samples. This must be a multiple of 1024.
		 * @return 
		 * 
		 */
		public function PlaybackEngine(dataProvider:IAudioDataProvider,playbackBufferSize:int=1024)
		{
			this.changingBufferSize=false;
			if((playbackBufferSize%1024)!=0)
			{
				throw new Error("playbackBufferSize must be multiples of 1024");
			}
			this.playbackBufferSize=playbackBufferSize;
			this.dataProvider=dataProvider;
			
			ready=false;
			soundStack=[];
			
			var fmt:PCMSoundFormat=this.dataProvider.getFormat();
			
			factory=new SoundFactory(fmt);
			
			factory.setBufferSize(playbackBufferSize);
			
			factory.addEventListener(SoundEvent.SOUND_OBJECT_CREATED,silentSoundHandler,false,0,true);
			var silentSoundSize:int=playbackBufferSize-1;
			
			factory.createSilentSoundObject(silentSoundSize);

		}
		/**
		 * Changes the buffer size of the playback engine in real time.
		 * @param newBufferSize The new buffer size. Must be a multiple of 1024
		 * 
		 */
		public function setBufferSize(newBufferSize:int):void
		{
			if(newBufferSize==playbackBufferSize)return;
			if((newBufferSize%1024)!=0)
			{
				throw new Error("newBufferSize must be multiples of 1024");
			}
			this.newBufferSize=newBufferSize;
			
			factory.setBufferSize(newBufferSize);
			
			this.changingBufferSize=true;
			
		}
		/**
		 * Handles the creation of a silent sound object. Called as a result of changing the buffer size.
		 * @param soundEvent The event emitted by the SoundFactory, carrying the newly generated sound.
		 * 
		 */
		private function silentSoundHandler(soundEvent:SoundEvent):void
		{
			silentSound=soundEvent.sound;
			factory.removeEventListener(SoundEvent.SOUND_OBJECT_CREATED,silentSoundHandler);
			factory.addEventListener(SoundEvent.SOUND_OBJECT_CREATED,newSound,false,0,true);
			factory.createSoundObject(dataProvider.getData(playbackBufferSize));
		}
		/**
		 * Handles the creation of a sound object.
		 * @param soundEvent The event emitted by the SoundFactory, carrying the newly generated sound.
		 * 
		 */
		private function newSound(soundEvent:SoundEvent):void
		{
			soundStack.push(soundEvent.sound);
			if(!ready)
			{
				ready=true;
				this.dispatchEvent(new Event(READY));
			}
		}
		/**
		 * Starts playback. This can be called any time after the PlaybackEngine has emitted a READY event.
		 * 
		 */
		public function start():void
		{
			if(!ready)
			{
				throw(new Error("PlaybackEngine is not yet ready. Wait for READY event"));
			}
			soundComplete(null);
		}
		/**
		 * Handler for the SOUND_COMPLETE event emitted by the channel playing the silent sound. This will be called upon multiples of 1024 samples. All timing is hinged off this event.
		 * @param event The SOUND_COMPLETE event.
		 * 
		 */
		private function soundComplete(event:Event):void
		{
			if(soundStack.length==0) //the data isn't ready yet... come back in a bit to try again
			{
				var t:Timer=new Timer(1,1);
				t.addEventListener(TimerEvent.TIMER_COMPLETE,soundComplete);
				t.start();
				return;			
			}
			var sound:Sound=soundStack.shift() as Sound;
			sound.play();
			var channel:SoundChannel=silentSound.play();
			channel.addEventListener(Event.SOUND_COMPLETE,soundComplete,false,0,true);
			
			if(!changingBufferSize)
			{
				factory.createSoundObject(dataProvider.getData(playbackBufferSize));
			}
			else
			{
				if(soundStack.length==0)
				{
					factory.removeEventListener(SoundEvent.SOUND_OBJECT_CREATED,newSound);
					factory.addEventListener(SoundEvent.SOUND_OBJECT_CREATED,silentSoundHandler,false,0,true);
					playbackBufferSize=newBufferSize;
					this.changingBufferSize=false;
					var silentSoundSize:int=playbackBufferSize-1;
					factory.createSilentSoundObject(silentSoundSize);

				}
			}
			
			if(Debug.ON)Debug.forceGC();						

		}

	}
}