package nu.mine.flashnet.sound.core
{
	import flash.utils.ByteArray;
	import flash.utils.Endian;
	import flash.events.EventDispatcher;
	import flash.display.Loader;
	import flash.media.Sound;
	import flash.events.Event;

	/**
	 * This class is the core of the audio library. 
	 * It is reponsible for taking a ByteArray and wrapping it in the necessary SWF bytecode in another ByteArray, loading the resulting SWF from the ByteArray and grabbing the definition of "SoundClass" that exists in the generated SWF. 
	 * An instance of SoundClass is created and returned aysynchronously to the user in the SoundEvent.SOUND_OBJECT_CREATED event.
	 * @author spender
	 * 
	 */
	internal class SoundFactory extends EventDispatcher
	{
		private var busy:Boolean;
		private var soundFormat:PCMSoundFormat;
		private var loader:Loader;

		private var bufferStart:Number;
		
		private var audioSwf:ByteArray;
		private var silenceSwf:ByteArray;
		
		/**
		 * Creates a SoundFactory
		 * @param soundFormat Defines the format of audio data that will be passed to the factory
		 * @return 
		 * 
		 */
		public function SoundFactory(soundFormat:PCMSoundFormat)
		{
			this.busy=false;
			this.soundFormat=soundFormat;

			loader=new Loader();
			loader.contentLoaderInfo.addEventListener(Event.COMPLETE,swfCreated,false,0,true);
		}
		
		/**
		 * @param bufferSize	The size of the audio buffer :)
		 */
		public function setBufferSize(bufferSize:Number):void
		{
			// rebuild audio buffer swf
			audioSwf = buildBufferSwf(bufferSize);
		}
		/**
		 * Copies bytes stored in a standard Array into a ByteArray. No attempt is made to validate the source data.
		 * @param src The source Array
		 * @param dest The destination ByteArray
		 * 
		 */
		private function writeBytes(src:Array,dest:ByteArray):void
		{
			for(var i:int=0;i<src.length;++i)
			{
				dest.writeByte(src[i]);
			}
		}
		/**
		 * Requests the creation of a Sound object that contains silence. This is used by the playback engine to drive the timing.
		 * Makes use of createSoundObject. See createSoundObject for more info.
		 * @param samples The length of the silent sound in samples.
		 * 
		 */
		internal function createSilentSoundObject(samples:int):void
		{
			var ba:ByteArray=new ByteArray();
			var numOfBytes:int=samples*soundFormat.channels*soundFormat.sampleSize;
			var val:int=(soundFormat.sampleSize==1)?128:0;  //this took ages to find... remember, 8bit audio is unsigned, so zero point is 128.
			for(var i:int=0;i<numOfBytes;++i)
			{
				ba.writeByte(val);
			}
			
			silenceSwf = getSwfByteArray(ba);
			loader.loadBytes(silenceSwf);
		}
		/**
		 * Requests the creation of a sound object containing audio data supplied by the user.
		 * @param audioData Audio data. No attempt is made to validate this data. It is for the user to ensure that the data is the correct length (i.e. ends on a sample boundary)
		 * 
		 */
		internal function createSoundObject(audioData:ByteArray):Boolean
		{
			if(busy)
			{
				throw(new Error("busy"));
				return false;
			}
			busy=true;
			
			audioSwf.position = bufferStart;
			audioSwf.writeBytes(audioData);
			loader.loadBytes(audioSwf); //TODO: a potential bug here. what if the loader is already processing a loadBytes request??? perhaps a request queue should be implemented, or many loaders?
			
			return true;
		}
		/**
		 * Handler for the Loader COMPLETE event
		 * @param event A COMPLETE event
		 * 
		 */
		private function swfCreated(event:Event):void
		{
			busy=false;
			var soundClass:Class=Class(loader.contentLoaderInfo.applicationDomain.getDefinition("SoundClass"));
			var sound:Sound=new soundClass();
			var event:Event=new SoundEvent(SoundEvent.SOUND_OBJECT_CREATED,sound);
			dispatchEvent(event);
		}
		/**
		 * Method responsible for generating valid SWF bytecode from provided audio data. For more info about the format of the SWF, see SoundClassSwfByteCode
		 * @param audioData The audio data to be wrapped
		 * @return A ByteArray containing valid SWF bytecode
		 * 
		 */
		private function getSwfByteArray(audioData:ByteArray):ByteArray
		{
			var numSamples:int=audioData.length/(soundFormat.channels*soundFormat.sampleSize);
			var swfData:ByteArray=new ByteArray();
			swfData.endian=Endian.LITTLE_ENDIAN;
			writeBytes(SoundClassSwfByteCode.soundClassSwfBytes1,swfData);
			swfData.writeUnsignedInt(299+numSamples*soundFormat.sampleSize*soundFormat.channels);
			writeBytes(SoundClassSwfByteCode.soundClassSwfBytes2,swfData);
			swfData.writeUnsignedInt(numSamples*soundFormat.channels*soundFormat.sampleSize+7);
			swfData.writeByte(1);
			swfData.writeByte(0);
			var formatByte:int=0x30;
			formatByte+=((soundFormat.sampleRateIndex<<2)+((soundFormat.sampleSize-1)<<1)+(soundFormat.channels-1));
			swfData.writeByte(formatByte);		
			swfData.writeUnsignedInt(numSamples);
			swfData.writeBytes(audioData);
			writeBytes(SoundClassSwfByteCode.soundClassSwfBytes3,swfData);
			return swfData;
		}
		/**
		 * Instead of building a complete swf every cycle, just create one swf for this buffer size
		 * Store the start position of the audio data and how long it is, then just refill that
		 * part.
		 */
		private function buildBufferSwf(numSamples:Number):ByteArray
		{
			var swfData:ByteArray=new ByteArray();
			swfData.endian=Endian.LITTLE_ENDIAN;
			
			writeBytes(SoundClassSwfByteCode.soundClassSwfBytes1,swfData);
			swfData.writeUnsignedInt(299+numSamples*soundFormat.sampleSize*soundFormat.channels);
			writeBytes(SoundClassSwfByteCode.soundClassSwfBytes2,swfData);
			swfData.writeUnsignedInt(numSamples*soundFormat.channels*soundFormat.sampleSize+7);
			swfData.writeByte(1);
			swfData.writeByte(0);
			var formatByte:int=0x30;
			formatByte+=((soundFormat.sampleRateIndex<<2)+((soundFormat.sampleSize-1)<<1)+(soundFormat.channels-1));
			swfData.writeByte(formatByte);		
			swfData.writeUnsignedInt(numSamples);
			
			// Store the position of the start of the audio data
			bufferStart = swfData.position;
			
			// Fill the buffer with zeros
			for(var i:Number=0;i<numSamples * soundFormat.sampleSize * soundFormat.channels;i++)
			{
				swfData.writeByte(0);
			}
			
			writeBytes(SoundClassSwfByteCode.soundClassSwfBytes3,swfData);
			
			return swfData;			
		}
	}
}